<?php
/**
 * Test PayFort Integration
 * 
 * This file tests the PayFort integration to ensure everything is working correctly.
 * Run this file to verify your configuration and API connectivity.
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

// Include required files
require_once 'payfort-config.php';
require_once 'gerar-pix.php';

// Test data
$testData = [
    'nome' => 'João da Silva',
    'cpf' => '12345678901',
    'email' => 'joao@exemplo.com',
    'telefone' => '11999999999',
    'valor' => 50.00,
    'descricao' => 'Teste de integração PayFort'
];

echo "=== TESTE DE INTEGRAÇÃO PAYFORT ===\n\n";

// Test 1: Configuration
echo "1. Testando configuração...\n";
try {
    validatePayFortConfig();
    echo "✓ Configuração válida\n";
} catch (Exception $e) {
    echo "✗ Erro na configuração: " . $e->getMessage() . "\n";
    exit(1);
}

// Test 2: API Key
echo "2. Testando API Key...\n";
if (isPayFortConfigured()) {
    echo "✓ API Key configurada\n";
} else {
    echo "✗ API Key não configurada\n";
    exit(1);
}

// Test 3: URLs
echo "3. Testando URLs...\n";
echo "   Cash-In URL: " . getPayFortCashInUrl() . "\n";
echo "   Cash-Out URL: " . getPayFortCashOutUrl() . "\n";
echo "   Webhook URL: " . getPayFortWebhookUrl() . "\n";

// Test 4: Generate PIX
echo "4. Testando geração de PIX...\n";
try {
    $result = gerarPix($testData);
    
    if ($result['success']) {
        echo "✓ PIX gerado com sucesso\n";
        echo "   Transaction ID: " . ($result['transaction_id'] ?? 'N/A') . "\n";
        echo "   External ID: " . ($result['external_id'] ?? 'N/A') . "\n";
        echo "   Order ID: " . ($result['order_id'] ?? 'N/A') . "\n";
        echo "   Status: " . ($result['status'] ?? 'N/A') . "\n";
        echo "   PIX Code: " . (isset($result['pix_code']) ? 'Gerado' : 'N/A') . "\n";
        
        // Test 5: Check transaction status
        if ($result['transaction_id']) {
            echo "5. Testando verificação de status...\n";
            $statusResult = verificarStatusTransacao($result['transaction_id']);
            
            if ($statusResult['success']) {
                echo "✓ Status verificado com sucesso\n";
                echo "   Status: " . ($statusResult['status'] ?? 'N/A') . "\n";
            } else {
                echo "✗ Erro ao verificar status: " . ($statusResult['error'] ?? 'Erro desconhecido') . "\n";
            }
        }
        
        // Test 6: Check by Order ID
        if ($result['order_id']) {
            echo "6. Testando verificação por Order ID...\n";
            $orderStatusResult = verificarStatusTransacaoPorOrderId($result['order_id']);
            
            if ($orderStatusResult['success']) {
                echo "✓ Status por Order ID verificado com sucesso\n";
                echo "   Status: " . ($orderStatusResult['status'] ?? 'N/A') . "\n";
            } else {
                echo "✗ Erro ao verificar status por Order ID: " . ($orderStatusResult['error'] ?? 'Erro desconhecido') . "\n";
            }
        }
        
    } else {
        echo "✗ Erro ao gerar PIX: " . ($result['error'] ?? 'Erro desconhecido') . "\n";
        
        if (isset($result['debug'])) {
            echo "   Debug info: " . json_encode($result['debug']) . "\n";
        }
    }
    
} catch (Exception $e) {
    echo "✗ Exceção ao gerar PIX: " . $e->getMessage() . "\n";
}

// Test 7: API Status
echo "7. Testando status da API...\n";
try {
    $response = file_get_contents('gerar-pix.php');
    $data = json_decode($response, true);
    
    if ($data && isset($data['status'])) {
        echo "✓ API respondendo\n";
        echo "   Status: " . $data['status'] . "\n";
        echo "   Service: " . ($data['service'] ?? 'N/A') . "\n";
        echo "   Version: " . ($data['version'] ?? 'N/A') . "\n";
        echo "   Provider: " . ($data['provider'] ?? 'N/A') . "\n";
    } else {
        echo "✗ API não está respondendo corretamente\n";
    }
} catch (Exception $e) {
    echo "✗ Erro ao testar API: " . $e->getMessage() . "\n";
}

echo "\n=== TESTE CONCLUÍDO ===\n";

// Return JSON response for web requests
if (isset($_SERVER['HTTP_ACCEPT']) && strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false) {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => true,
        'message' => 'Teste de integração PayFort concluído',
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?>
